// STVS background service worker (Manifest V3)
// Responsible for creating the downloadable UTF-8 text file via chrome.downloads.download().

chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
  if (message && message.type === "STVS_DOWNLOAD_TEXT") {
    const payload = message.payload || {};
    const fullText = payload.text;
    const title = payload.title || "";
    const url = payload.url || "";
    const domain = payload.domain || "";

    if (!fullText || typeof fullText !== "string") {
      sendResponse({ ok: false, error: "No text content provided for download." });
      return false;
    }

    const filename = buildFilename(title, domain);
    // Use a data URL so it works reliably from a service worker context.
    const dataUrl =
      "data:text/plain;charset=utf-8," + encodeURIComponent(fullText);

    chrome.downloads.download(
      {
        url: dataUrl,
        filename,
        saveAs: true,
        conflictAction: "uniquify"
      },
      downloadId => {
        if (chrome.runtime.lastError) {
          sendResponse({
            ok: false,
            error: chrome.runtime.lastError.message || "Download failed."
          });
        } else if (!downloadId && downloadId !== 0) {
          sendResponse({
            ok: false,
            error: "Download could not be started."
          });
        } else {
          sendResponse({ ok: true, downloadId });
        }
      }
    );

    // Keep the service worker alive while we complete the async download.
    return true;
  }

  return false;
});

function buildFilename(rawTitle, rawDomain) {
  const now = new Date();
  const year = String(now.getFullYear());
  const month = String(now.getMonth() + 1).padStart(2, "0");
  const day = String(now.getDate()).padStart(2, "0");
  const hours = String(now.getHours()).padStart(2, "0");
  const mins = String(now.getMinutes()).padStart(2, "0");

  const datePart = `${year}${month}${day}-${hours}${mins}`;

  const domain = (rawDomain || "").toLowerCase().replace(/^https?:\/\//, "");

  const sanitizedTitle = sanitizeForFilename(rawTitle || "article");

  // Base filename before enforcing max length.
  let base = `STVS_${datePart}_${domain || "source"}_${sanitizedTitle}`;

  // Enforce maximum length of 120 characters including extension.
  const maxLength = 120;
  const extension = ".txt";
  const maxBaseLength = maxLength - extension.length;

  if (base.length > maxBaseLength) {
    base = base.slice(0, maxBaseLength);
  }

  return `${base}${extension}`;
}

function sanitizeForFilename(name) {
  // Remove characters that are illegal in filenames on major OSes.
  let sanitized = (name || "")
    .replace(/[\/\\:\*\?"<>\|]/g, " ")
    .replace(/\s+/g, " ")
    .trim();

  if (!sanitized) {
    sanitized = "article";
  }

  // Use underscores for spaces to keep it compact.
  sanitized = sanitized.replace(/ /g, "_");

  return sanitized;
}

