// STVS popup script

document.addEventListener("DOMContentLoaded", () => {
  const exportButton = document.getElementById("stvs-export");
  const statusEl = document.getElementById("stvs-status");
  const progressBar = document.getElementById("stvs-progress");

  const coldHintEl = document.getElementById("stvs-cold-hint");
  const contactTrigger = document.getElementById("stvs-contact-trigger");
  const STORAGE_KEY = "stvsLastBackendUse";
  // Render free tier: spins down after 15 min without traffic (https://render.com/docs/free#spinning-down-on-idle)
  const IDLE_MS = 15 * 60 * 1000;
  const COLD_HINT_TEXT = "When the AI hasn’t been used in over 15 minutes, the first request may take up to 30 seconds.";
  const NORMAL_WAIT_HINT = "This may take a few seconds.";

  // Tooltip: Contact us only (~1s delay)
  const tooltipContact = document.getElementById("stvs-tooltip-contact");
  let contactTooltipTimer = null;
  if (tooltipContact && contactTrigger) {
    contactTrigger.addEventListener("mouseenter", () => {
      contactTooltipTimer = setTimeout(() => {
        tooltipContact.classList.add("stvs-tooltip--visible");
        tooltipContact.setAttribute("aria-hidden", "false");
      }, 800);
    });
    contactTrigger.addEventListener("mouseleave", () => {
      if (contactTooltipTimer) clearTimeout(contactTooltipTimer);
      contactTooltipTimer = null;
      tooltipContact.classList.remove("stvs-tooltip--visible");
      tooltipContact.setAttribute("aria-hidden", "true");
    });
  }

  // Contact: open feedback in a separate popup window
  if (contactTrigger) {
    contactTrigger.addEventListener("click", (e) => {
      e.preventDefault();
      chrome.tabs.query({ active: true, currentWindow: true }, (tabs) => {
        let url = "";
        if (tabs && tabs[0] && tabs[0].url && !tabs[0].url.startsWith("chrome://")) {
          url = tabs[0].url;
        }
        const feedbackUrl = chrome.runtime.getURL("feedback.html") + "?url=" + encodeURIComponent(url);
        chrome.windows.create({
          url: feedbackUrl,
          type: "popup",
          width: 460,
          height: 620
        });
      });
    });
  }

  function setStatus(message, type) {
    statusEl.textContent = message || "";
    statusEl.classList.remove("stvs-status--error", "stvs-status--success");
    if (type === "error") {
      statusEl.classList.add("stvs-status--error");
    } else if (type === "success") {
      statusEl.classList.add("stvs-status--success");
    }
  }

  const exportTextEl = document.getElementById("stvs-export-text");
  const progressContainer = progressBar ? progressBar.parentElement : null;
  let slowExportTimer = null;
  let slowExportInterval = null;
  const SLOW_EXPORT_MS = 5000;

  function clearSlowExportTimers() {
    if (slowExportTimer) {
      clearTimeout(slowExportTimer);
      slowExportTimer = null;
    }
    if (slowExportInterval) {
      clearInterval(slowExportInterval);
      slowExportInterval = null;
    }
  }

  function setLoading(isLoading) {
    exportButton.disabled = isLoading;
    if (exportTextEl) {
      exportTextEl.textContent = isLoading ? "Exporting..." : "Export";
    }
    if (progressContainer) {
      if (isLoading) {
        progressContainer.classList.add("stvs-progress-container--visible");
      } else {
        progressContainer.classList.remove("stvs-progress-container--visible");
      }
    }
    if (!isLoading) {
      clearSlowExportTimers();
    }
  }

  function updateProgress(percent, message) {
    if (progressBar) {
      progressBar.style.width = `${percent}%`;
      progressBar.setAttribute("aria-valuenow", percent);
    }
    if (message && !slowExportInterval) {
      setStatus(message, null);
    }
    if (coldHintEl) {
      if (message === "Cleaning content with AI...") {
        chrome.storage.local.get(STORAGE_KEY, (result) => {
          const last = result && result[STORAGE_KEY];
          const isColdStart = !last || (Date.now() - last) > IDLE_MS;
          coldHintEl.textContent = isColdStart ? COLD_HINT_TEXT : NORMAL_WAIT_HINT;
          coldHintEl.style.display = "block";
        });
      } else {
        coldHintEl.style.display = "none";
      }
    }
  }

  function resetProgress() {
    if (progressBar) {
      progressBar.style.width = "0%";
      progressBar.setAttribute("aria-valuenow", 0);
    }
    if (coldHintEl) coldHintEl.style.display = "none";
  }

  exportButton.addEventListener("click", () => {
    chrome.tabs.query({ active: true, currentWindow: true }, tabs => {
      const tab = tabs && tabs[0];
      if (!tab || !tab.id) {
        setStatus("No active tab found.", "error");
        return;
      }

      setLoading(true);
      resetProgress();
      setStatus("Extracting article...", null);
      updateProgress(10, "Extracting article...");

      slowExportTimer = setTimeout(() => {
        slowExportTimer = null;
        if (!exportButton.disabled) return;
        chrome.storage.local.get(STORAGE_KEY, (result) => {
          const last = result && result[STORAGE_KEY];
          const isColdStart = !last || (Date.now() - last) > IDLE_MS;
          if (isColdStart) return;
          if (!exportButton.disabled) return;
          const dots = [".", "..", "..."];
          let i = 0;
          setStatus("Still exporting" + dots[0], null);
          slowExportInterval = setInterval(() => {
            if (!exportButton.disabled) {
              clearSlowExportTimers();
              return;
            }
            i = (i + 1) % 3;
            setStatus("Still exporting" + dots[i], null);
          }, 400);
        });
      }, SLOW_EXPORT_MS);

      // Ensure content scripts are injected for this tab before messaging.
      chrome.scripting.executeScript(
        {
          target: { tabId: tab.id },
          files: ["readability.js", "contentScript.js"]
        },
        (results) => {
          if (chrome.runtime.lastError) {
            setLoading(false);
            resetProgress();
            const errorMsg = chrome.runtime.lastError.message || "";
            console.error("Script injection error:", errorMsg);
            // Check if it's a chrome:// or extension page
            if (tab.url && (tab.url.startsWith("chrome://") || tab.url.startsWith("chrome-extension://") || tab.url.startsWith("edge://"))) {
              setStatus("This page cannot be processed. Please run STVS on a normal web article page (http/https).", "error");
            } else {
              setStatus(`Script injection failed: ${errorMsg}. Please reload the page and try again.`, "error");
            }
            return;
          }

          updateProgress(30, "Analyzing content...");

          chrome.tabs.sendMessage(
            tab.id,
            { type: "STVS_EXTRACT" },
            async response => {
              if (chrome.runtime.lastError) {
                setLoading(false);
                resetProgress();
                const msg = chrome.runtime.lastError.message || "";
                if (msg.includes("Receiving end does not exist")) {
                  setStatus("This page cannot be processed. Please run STVS on a normal web article page (http/https).", "error");
                } else {
                  setStatus("Extraction failed. Try again on a different page.", "error");
                }
                return;
              }

              if (!response || !response.ok || !response.data) {
                setLoading(false);
                resetProgress();
                const errorDetail = response && response.error ? ` Reason: ${response.error}` : "";
                setStatus(`Extraction failed. Try again.${errorDetail}`, "error");
                return;
              }

              const { title, content: rawContent, domSummary, url, domain, publishedDate, modifiedDate } = response.data;

              let finalContent = rawContent;
              updateProgress(50, "Cleaning content with AI...");

              try {
                const result = await callStvsBackendExtractAndReview({
                  domSummary: domSummary || rawContent,
                  url,
                  title
                });
                if (result.content && result.content.length > 0) {
                  finalContent = result.content;
                  try {
                    // If AI says this doesn't look like article body, get DOM and re-extract by AI instructions
                    if (result.needDom) {
                      updateProgress(72, "Analyzing page structure...");
                      const domResponse = await new Promise((resolve) => {
                        chrome.tabs.sendMessage(tab.id, { type: "STVS_GET_DOM" }, resolve);
                      });
                      if (chrome.runtime.lastError || !domResponse || !domResponse.ok) {
                        console.warn("STVS get DOM failed:", chrome.runtime.lastError || domResponse);
                      } else {
                        try {
                          const instructions = await callStvsBackendAnalyzeDom({
                            dom: domResponse.dom,
                            url,
                            title
                          });
                          if (instructions) {
                            updateProgress(75, "Re-extracting with AI instructions...");
                            const selResponse = await new Promise((resolve) => {
                              chrome.tabs.sendMessage(tab.id, {
                                type: "STVS_EXTRACT_WITH_SELECTORS",
                                payload: {
                                  rootSelector: instructions.rootSelector,
                                  paragraphSelector: instructions.paragraphSelector,
                                  excludeSelectors: instructions.excludeSelectors || []
                                }
                              }, resolve);
                            });
                            if (!chrome.runtime.lastError && selResponse && selResponse.ok && selResponse.content && selResponse.content.length >= 100) {
                              updateProgress(78, "Cleaning re-extracted content...");
                              const reReviewed = await callStvsBackendReview({
                                content: selResponse.content,
                                url,
                                title
                              });
                              if (reReviewed.content && reReviewed.content.length > 0) {
                                finalContent = reReviewed.content;
                              } else {
                                finalContent = selResponse.content;
                              }
                            }
                          }
                        } catch (domErr) {
                          console.warn("STVS DOM fallback error:", domErr);
                        }
                      }
                    }
                  } catch (reviewErr) {
                    // If review fails, use the cleaned content from first pass
                    console.warn("STVS review error:", reviewErr);
                  }
                }
                updateProgress(85, "Preparing file...");
              } catch (err) {
                // If the backend fails, fall back to the locally extracted content.
                console.warn("STVS backend error:", err);
                updateProgress(70, "Using local extraction...");
              }

              const fileText = buildFileText({ title, content: finalContent, url, domain });

              updateProgress(90, "Downloading...");

              chrome.runtime.sendMessage(
                {
                  type: "STVS_DOWNLOAD_TEXT",
                  payload: {
                    text: fileText,
                    title: title,
                    url: url,
                    domain: domain
                  }
                },
                downloadResponse => {
                  setLoading(false);
                  updateProgress(100, "Complete!");

                  if (chrome.runtime.lastError || !downloadResponse) {
                    resetProgress();
                    const detail = chrome.runtime.lastError
                      ? ` Reason: ${chrome.runtime.lastError.message}`
                      : "";
                    setStatus(`Download failed. Please try again.${detail}`, "error");
                    return;
                  }

                  if (!downloadResponse.ok) {
                    resetProgress();
                    const detail = downloadResponse.error
                      ? ` Reason: ${downloadResponse.error}`
                      : "";
                    setStatus(`Download failed. Please try again.${detail}`, "error");
                    return;
                  }

                  setStatus("Exported. Choose where to save the file.", "success");
                  chrome.storage.local.set({ [STORAGE_KEY]: Date.now() });
                  // Reset progress bar after a short delay
                  setTimeout(() => {
                    resetProgress();
                  }, 2000);
                }
              );
            }
          );
        }
      );
    });
  });

  function buildFileText({ title, content, url, domain }) {
    const safeTitle = (title || "").trim() || "Untitled";
    const safeUrl = url || "";
    const safeDomain = domain || "";

    const lines = [
      `Title: ${safeTitle}`,
      `Source: ${safeDomain}`,
      `URL: ${safeUrl}`,
      "---",
      "",
      content || ""
    ];

    return lines.join("\n");
  }

  // Single backend call: extract + review in one request to reduce latency.
  // Returns { content, needDom }. On failure, caller should fall back to local extraction.
  async function callStvsBackendExtractAndReview({ domSummary, url, title }) {
    const backendUrl = "https://stvs-backend.onrender.com/extract-and-review";

    const resp = await fetch(backendUrl, {
      method: "POST",
      headers: {
        "Content-Type": "application/json"
      },
      body: JSON.stringify({
        domSummary,
        url,
        title
      })
    });

    if (!resp.ok) {
      throw new Error(`Backend HTTP ${resp.status}`);
    }

    const data = await resp.json();
    if (!data || !data.ok || typeof data.content !== "string") {
      throw new Error(data && data.error ? data.error : "Invalid backend response");
    }

    return {
      content: data.content.trim(),
      needDom: !!data.needDom
    };
  }

  // Review-only (used for needDom re-extraction path). Returns { content, needDom }.
  async function callStvsBackendReview({ content, url, title }) {
    const backendUrl = "https://stvs-backend.onrender.com/review";

    const resp = await fetch(backendUrl, {
      method: "POST",
      headers: {
        "Content-Type": "application/json"
      },
      body: JSON.stringify({
        content,
        url,
        title
      })
    });

    if (!resp.ok) {
      throw new Error(`Backend HTTP ${resp.status}`);
    }

    const data = await resp.json();
    if (!data || !data.ok || typeof data.content !== "string") {
      throw new Error(data && data.error ? data.error : "Invalid backend response");
    }

    return {
      content: data.content.trim(),
      needDom: !!data.needDom
    };
  }

  // When AI says content is not article body, send DOM and get extraction instructions.
  async function callStvsBackendAnalyzeDom({ dom, url, title }) {
    const backendUrl = "https://stvs-backend.onrender.com/analyze-dom";

    const resp = await fetch(backendUrl, {
      method: "POST",
      headers: {
        "Content-Type": "application/json"
      },
      body: JSON.stringify({ dom, url, title })
    });

    if (!resp.ok) {
      throw new Error(`Backend HTTP ${resp.status}`);
    }

    const data = await resp.json();
    if (!data || !data.ok) {
      throw new Error(data && data.error ? data.error : "Invalid analyze-dom response");
    }

    return {
      rootSelector: data.rootSelector,
      paragraphSelector: data.paragraphSelector,
      excludeSelectors: data.excludeSelectors || []
    };
  }
});
