// Lightweight Readability-style extractor.
// NOTE: This is a simplified implementation that exposes a Readability class
// with a parse() method returning { title, textContent }.
// It operates on a cloned document and does not mutate the live page.

(function (global) {
  function Readability(doc, options) {
    if (!doc || !doc.documentElement) {
      throw new Error("Readability: first argument must be a Document.");
    }
    this._doc = doc;
    this._options = options || {};
  }

  Readability.prototype.parse = function () {
    const doc = this._doc;
    const title = (doc.title || "").trim() || inferTitleFromDocument(doc) || "";
    const mainNode = findBestArticleNode(doc);
    const textContent = mainNode ? extractTextFromNode(mainNode) : "";

    return {
      title,
      textContent
    };
  };

  function inferTitleFromDocument(doc) {
    const h1 = doc.querySelector("h1");
    if (h1 && h1.textContent) {
      return h1.textContent.trim();
    }
    const ogTitle = doc.querySelector('meta[property="og:title"], meta[name="og:title"]');
    if (ogTitle && ogTitle.content) {
      return ogTitle.content.trim();
    }
    const twitterTitle = doc.querySelector('meta[name="twitter:title"]');
    if (twitterTitle && twitterTitle.content) {
      return twitterTitle.content.trim();
    }
    return "";
  }

  function findBestArticleNode(doc) {
    // Prefer explicit semantic containers first.
    const primary = doc.querySelector("article, main");
    if (primary) {
      return primary;
    }
    // Otherwise, fall back to density-scoring heuristic.
    const candidates = doc.querySelectorAll("article, main, section, div");
    let bestNode = null;
    let bestScore = 0;

    const negativePatterns = [
      "comment",
      "comments",
      "footer",
      "footnote",
      "header",
      "nav",
      "menu",
      "sidebar",
      "aside",
      "promo",
      "advert",
      "ad-",
      "sponsor",
      "social",
      "subscribe",
      "popup",
      "modal"
    ];
    const positivePatterns = [
      "article",
      "content",
      "post",
      "story",
      "main",
      "body",
      "entry",
      "page"
    ];

    candidates.forEach(node => {
      if (!(node instanceof HTMLElement)) return;

      const text = (node.innerText || "").trim();
      const length = text.length;
      if (length < 400) return; // ignore very small blocks

      const links = node.querySelectorAll("a");
      let linkTextLength = 0;
      links.forEach(a => {
        linkTextLength += (a.innerText || "").length;
      });
      const linkDensity = length > 0 ? linkTextLength / length : 0;

      // base score by text length and link density
      let score = length * (1 - linkDensity);

      const classId = ((node.className || "") + " " + (node.id || "")).toLowerCase();
      if (negativePatterns.some(p => classId.includes(p))) {
        score *= 0.3;
      }
      if (positivePatterns.some(p => classId.includes(p))) {
        score *= 1.2;
      }

      if (score > bestScore) {
        bestScore = score;
        bestNode = node;
      }
    });

    return bestNode;
  }

  function extractTextFromNode(node) {
    // Build paragraphs from visible text-like elements.
    const blocks = [];
    const seen = new Set();

    const walker = document.createTreeWalker(
      node,
      NodeFilter.SHOW_ELEMENT,
      {
        acceptNode: function (el) {
          const tag = el.tagName;
          if (!tag) return NodeFilter.FILTER_SKIP;
          if (/^(SCRIPT|STYLE|NOSCRIPT|IFRAME|SVG|CANVAS|FORM|HEADER|FOOTER|NAV|ASIDE|FIGURE|FIGCAPTION)$/i.test(tag)) {
            return NodeFilter.FILTER_REJECT;
          }
          // Only treat leaf text blocks as paragraphs to avoid duplication:
          if (/^(P|H1|H2|H3|H4|H5|H6|LI|BLOCKQUOTE|PRE)$/i.test(tag)) {
            return NodeFilter.FILTER_ACCEPT;
          }
          return NodeFilter.FILTER_SKIP;
        }
      },
      false
    );

    let current;
    while ((current = walker.nextNode())) {
      // Skip captions inside figures or figcaptions.
      if (current.closest && current.closest("figure, figcaption")) {
        continue;
      }
      const text = (current.innerText || "").trim();
      if (!text) continue;
      // Skip typical short photograph credits / captions.
      if (/^photograph:/i.test(text) && text.length < 200) {
        continue;
      }
      // De-duplicate identical blocks.
      if (seen.has(text)) continue;
      seen.add(text);
      blocks.push(text);
    }

    if (!blocks.length) {
      return (node.innerText || "").trim();
    }

    const raw = blocks.join("\n\n");
    return normalizeWhitespace(raw);
  }

  function normalizeWhitespace(text) {
    if (!text) return "";
    return text
      .replace(/\r\n/g, "\n")
      .replace(/\r/g, "\n")
      .replace(/[ \t]+/g, " ")
      .replace(/\n{3,}/g, "\n\n")
      .replace(/^[ \t]+/gm, "")
      .trim();
  }

  if (typeof module !== "undefined" && module.exports) {
    module.exports = Readability;
  } else {
    global.Readability = Readability;
  }
})(typeof window !== "undefined" ? window : this);

